# vim: set fileencoding=utf-8 :
#
#  - Copyright 2014 Guido Günther <agx@sigxcpu.org>
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

from unittest import mock

import pytest

from jenkins_jobs.config import JJBConfig
import jenkins_jobs.builder


_plugins_info = {}
_plugins_info["plugin1"] = {"longName": "", "shortName": "", "version": ""}


@pytest.fixture
def jjb_config():
    config = JJBConfig()
    config.validate()
    return config


def test_plugins_list(jjb_config):
    jjb_config.builder["plugins_info"] = _plugins_info

    builder = jenkins_jobs.builder.JenkinsManager(jjb_config)
    assert builder.plugins_list == _plugins_info


def test_plugins_list_from_jenkins(mocker, jjb_config):
    mocker.patch.object(
        jenkins_jobs.builder.jenkins.Jenkins, "get_plugins", return_value=_plugins_info
    )
    # Trigger fetching the plugins from jenkins when accessing the property
    jjb_config.builder["plugins_info"] = None
    builder = jenkins_jobs.builder.JenkinsManager(jjb_config)
    assert list(builder.plugins_list) == list(_plugins_info.values())


def test_delete_managed(mocker, jjb_config):
    jjb_config.builder["plugins_info"] = None
    builder = jenkins_jobs.builder.JenkinsManager(jjb_config)

    patches = mocker.patch.multiple(
        "jenkins_jobs.builder.JenkinsManager",
        get_jobs=mock.DEFAULT,
        is_job=mock.DEFAULT,
        is_managed=mock.DEFAULT,
        delete_job=mock.DEFAULT,
    )
    patches["get_jobs"].return_value = [
        {"fullname": "job1"},
        {"fullname": "job2"},
    ]
    patches["is_managed"].side_effect = [True, True]
    patches["is_job"].side_effect = [True, True]

    builder.delete_old_managed()
    assert patches["delete_job"].call_count == 2


@pytest.mark.parametrize("error_string", ["Connection refused", "Forbidden"])
def test_get_plugins_info_error(mocker, jjb_config, error_string):
    builder = jenkins_jobs.builder.JenkinsManager(jjb_config)
    exception = jenkins_jobs.builder.jenkins.JenkinsException(error_string)
    mocker.patch.object(builder.jenkins, "get_plugins", side_effect=exception)
    plugins_info = builder.get_plugins_info()
    assert [_plugins_info["plugin1"]] == plugins_info
